
from item_model import Actor, Room, Thing
from action_model import Behave, Modify, Sense, Configure
import can
import when

#TODO context actions
#TODO if no actions included, say so

discourse = {

    'metadata': {

        'title': 'The Singular Adventure of the Indigo Violet',

        'headline': 'From the reminiscences of John H. Watson',

        'people': [('adapted by', 'David Fletcher')],

        'prologue': """
"I do not think, Watson," said my friend, "that you have ever placed
before the public the facts of the case we investigated in the year
18--, in which the indigo violet was so deeply concerned."

"Indeed not," I replied.

"The case was a simple one, of course, but --"

"Simple, Holmes!" I expostulated.  "Really I do not see how it could
be called simple."

"Exceptionally so in one respect, Watson: that only the killer was
discovered to have been lying. The other witnesses recounted the facts
with a pleasing exactitude and honesty."

I recalled that Holmes, after a brief examination of the scene, had
asked the persons concerned only about each other and about those few rooms and
items that were involved, and yet by the application of his remarkable
deductive faculty had soon been able to accuse the guilty person.
"""},

    'spin': {
        'narrator': '@watson',
        'commanded': '@holmes',
        'focalizer': '@holmes',
        'speed': 0.75,
        'time': 'after' },

    'command_grammar': {
        'ASK ACCESSIBLE STRING': ['ask ACCESSIBLE about STRING',
                                  'ask ACCESSIBLE about STRING STRING',
                                  'ask ACCESSIBLE about STRING STRING STRING',
                                  'ask ACCESSIBLE about STRING STRING STRING STRING',
                                  'ask ACCESSIBLE about STRING STRING STRING STRING STRING' ],
        'ACCUSE ACCESSIBLE': ['accuse ACCESSIBLE' ]
        }
}

class Cosmos(Actor):

    def __init__(self):
        Actor.__init__( self,
                        '@cosmos',
                        called='creation',
                        referring=None,
                        allowed=can.have_any_item )
        self.current_narrator = '@watson'
        self.current_focalizer = '@holmes'
        self.current_narratee = None

        self.transient_narrator = None
        self.transient_focalizer = None
        self.transient_narratee = None

    def update_spin(self, world, discourse):

        if self.transient_narrator != None:
            discourse.spin[ 'narrator' ] = self.transient_narrator
            discourse.spin[ 'speed' ] = 1.0  # suppresses room titles
            discourse.spin[ 'window' ] = 999999
        else:
            discourse.spin[ 'narrator' ] = self.current_narrator
            discourse.spin[ 'speed' ] = 0.75
            discourse.spin[ 'window' ] = 'current'

        if self.transient_focalizer != None:
            discourse.spin[ 'focalizer' ] = self.transient_focalizer
        else:
            discourse.spin[ 'focalizer' ] = self.current_focalizer

        if self.transient_narratee != None:
            discourse.spin[ 'narratee' ] = self.transient_narratee
        else:
            discourse.spin[ 'narratee' ] = self.current_narratee

        self.transient_narrator = None
        self.transient_narratee = None
        self.transient_focalizer = None

        return discourse.spin

    def react(self, world, basis):

        if ( basis.modify and basis.direct == str(self) ):

             if (basis.feature == 'current_narrator' ):
                 self.current_narrator = basis.new_value

             elif ( basis.feature == 'transient_narrator' ):
                 self.transient_narrator = basis.new_value

             elif ( basis.feature == 'transient_narratee' ):
                 self.transient_narratee = basis.new_value

             elif ( basis.feature == 'transient_focalizer' ):
                 self.transient_focalizer = basis.new_value

        return []


cosmos = Cosmos()

def action_is_move_carrying( item, action ):
    return hasattr( action, 'moving_with' ) and item in action.moving_with

def action_involves( item, action ):
    return ( action.agent == item
             or ( hasattr( action, 'target' ) and action.target == item )
             or ( hasattr( action, 'direct' ) and action.direct == item )
             or ( hasattr( action, 'indirect' ) and action.indirect == item )
             or action_is_move_carrying( item, action ) )

def action_is_leave_by( actor, action ):
    return action.verb == 'leave' and action.agent == actor


def set_saliences( world, narrator, focalizer, topobj ):
    murder_time = 9999999
    for (id, a) in world.concept[ focalizer ].act.items():
        if a.verb == 'murderend': murder_time = a.start

    for (id, a) in world.concept[ focalizer ].act.items():

        if a.start >= murder_time:
            a.salience = 0.0

        elif hasattr( a, 'is_murdery' ) and a.is_murdery and narrator != '@holmes':
            a.salience = 0.0

        elif hasattr( a, 'include_for_focalizer' ):
            if a.include_for_focalizer( focalizer, topobj ):
                a.salience = 0.75
            else:
                a.salience = 0.0

        elif hasattr( a, 'include_for_narrator' ):
            if a.include_for_narrator( narrator, topobj ):
                a.salience = 0.75
            else:
                a.salience = 0.0

        elif action_involves( topobj, a ):
            a.salience = 0.75

        else:
            a.salience = 0.0

def set_narr_actions( narr, foc, nee ):
    return [ Modify( 'change_transient_narrator',
                     '@cosmos',
                     direct='@cosmos',
                     feature='transient_narrator',
                     new = narr ),
             Modify( 'change_transient_narratee',
                     '@cosmos',
                     direct='@cosmos',
                     feature='transient_narratee',
                     new = nee ),
             Modify( 'change_transient_focalizer',
                     '@cosmos',
                     direct='@cosmos',
                     feature='transient_focalizer',
                     new = foc ) ]

asks_count = 0

class Suspect(Actor):
    def __init__( self, tag_and_parent, **kw ):
        Actor.__init__( self, tag_and_parent, allowed = can.have_any_item, **kw )

    def react(self, world, basis):

        if ( basis.behave
             and basis.verb == 'ask'
             and basis.target == str(self) ):
            
            global asks_count
            asks_count += 1

            set_saliences( world, str( self ), str( self ), basis.topobj )

            return set_narr_actions( str( self ), str( self ), None )
        return []

def insufficient_data( world ):
    return True

when.insufficient_data = insufficient_data

class Holmes(Actor):
    def __init__( self ):
        Actor.__init__(
            self,
            '@holmes in @morningroom',
            called='Holmes',
            allowed=can.possess_any_thing,
            referring='| sherlock holmes',
            qualities=['person', 'man'],
            gender='male' )

    def prevent(self, world, basis):
        if basis.verb == 'take': return True
        return False

    def react(self, world, basis):
        if basis.behave and basis.verb == 'accuse':
            
            global asks_count
            if asks_count < 3:
                return [ Behave( 'wont_accuse',
                                 '@holmes',
                                 template = "it is a capital mistake to theorise without data" ) ]

            accused = basis.target
            if accused in suspect_tags:
                
                set_saliences( world, '@holmes', accused, accused )

                set_narr = set_narr_actions( '@holmes', accused, accused )

                if basis.target == '@flounce':
                    solve = Behave( 'solve',
                                    '@holmes',
                                    template = 'Holmes had solved the case!' )
                    solve.final = True
                    return set_narr + [ solve ]
                else:
                    doh = Behave( 'holmes_mistake',
                                  '@holmes',
                                  template = "I fear I am mistaken -- you cannot have been guilty. I must apologise." )
                    return set_narr + [ doh ]

        return []

class Watson(Actor):
    def __init__( self ):
        Actor.__init__( self,
                        '@watson in @morningroom',
                        called='Watson',
                        referring='| john watson',
                        qualities=['person', 'man'],
                        gender='male' )

    def react(self, world, basis):
        if ( basis.behave
             and basis.verb == 'leave'
             and hasattr( basis, 'direct' )
             and basis.direct == '@holmes' ):
            return [ Behave( 'leave',
                             '@watson',
                             direct = '@watson',
                             direction = basis.direction,
                             template = '[@watson/s] [follow/v] [@holmes/o]' ) ]

        return []


items = [

    Watson(),
    Holmes(),

    Room('@hall',
         article='the',
         called='hall',
         referring='| hall',
         sight="""
         [*/s] [is/v] in the hall.  The morning room [is/v] west and the library east""",
         exits={ 'west': '@morningroom',
                 'east': '@library' } ),

    Room('@morningroom',
         article='the',
         called='morning room',
         referring='| morning room',
         sight="""
         [*/s] [is/v] in the morning room.  The only exit [is/v] east""",
         exits={'east': '@hall' } ),

    Room('@library',
         article='the',
         called='library',
         referring='| library',
         sight="""
         [*/s] [is/v] in the library.  The study, where the murder was committed,
         [is/v] north.  The hall was to the west, and the drawing room to the east""",
         exits={'east': '@drawingroom',
                'north': '@study',
                'west': '@hall'} ),

    Room('@drawingroom',
         article='the',
         called='drawing room',
         referring='| drawing room',
         sight="""
         [*/s] [is/v] in the drawing room.  The library [is/v] west""",
         exits={'west': '@library'} ),

    Room('@study',
         article='the',
         called='study',
         referring='| study',
         sight="""
         [*/s] [is/v] in the study""",
         exits={'south': '@library' } ),

    Thing( '@salver in @hall',
           article='the',
           called='salver',
           referring='| salver' ),

    Thing( '@indigo in @library',
           article='the',
           called='indigo violet',
           referring='| indigo purple violet',
           sight="""the crushed remains of the indigo violet
                  had been left here by the murderer, after that flower's
                  deadly fumes had done their work""" ),

    Thing( '@victim in @study',
           article='the',
           called='body of Sir Horace Spigot',
           referring='| corpse body of sir horace spigot',
           gender='male',
           sight="""the hideous purplish tinge of the corpse's skin
                    clearly [indicate/v] a death caused by an indigo violet.""" ),

    Suspect('@flounce in @library',
            called='Miss Flounce',
            referring='| miss flounce',
            qualities=['person', 'woman'],
            gender='female',
            sight = """[*/s] [see/v] instantly from her mode of dress that [@flounce/s] [is/v]
                       governess to the Spigot family.""" ),

    Suspect('@mullion in @library',
            called='Lady Mullion',
            referring='| lady mullion',
            qualities=['person', 'woman'],
            gender='female',
            sight = """[*/s] [observe/v] that [@mullion/s] [is/v]
                       a respectable woman of middle age.""" ),

    Suspect('@trivet in @hall',
            called='Trivet',
            referring='| trivet butler',
            qualities=['person', 'man'],
            gender='male',
            sight = """[*/s] [observe/v] the peculiar indentations in the hands made by
                    carrying salvers, and [deduce/v] that [@trivet/s] [is/v] the butler.""" ),

    Suspect('@horsebrass in @library',
            called='Captain Horsebrass',
            referring='| capt captain horsebrass',
            qualities=['person', 'man'],
            gender='male',
            sight = """[*/s] [deduce/v] from his
                    manner and bearing that [@horsebrass/s] [is/v] a captain of the
                    Indigo Hussars, recently returned from the Trobriand Islands""" )
]

tag_to_called = dict( [ ( str( i ), i.article + ' ' + i.called[1] ) for i in items ] )

def find_matching_obj( toks ):
    tokset = set( toks )
    for i in items:
        (before, nouns, after) = i.referring
        allitemwords = before.union( nouns ).union( after )
        if tokset.issubset( allitemwords ): return i

    return None

class MyAsk(Behave):
    def __init__( self, agent, topobj, **kw ):
        Behave.__init__( self, 'ask', agent, **kw )
        self.topobj = topobj

def COMMAND_ask(agent, tokens, _):
    'to ask something about a topic'

    topobj = find_matching_obj( tokens[2:] )

    if tokens[2] == 'herself' or tokens[2] == 'himself':
        topobj = tokens[1]

    if tokens[2] == 'me' or tokens[2] == 'myself':
        topobj = agent

    if topobj is None:
        return None

    return MyAsk( agent, topobj,
                  target = tokens[1],
                  utterance = str( topobj ),
                  template='[agent/s] [ask/v] [direct/o] about [utterance/o]',
                  salience = 0 )

def COMMAND_accuse( agent, tokens, _):
    'to accuse something'

    return Behave( 'accuse',
                   agent,
                   target = tokens[1],
                   template='[agent/s] [accuse/v] [direct/o]',
                   salience = 0 )

suspect_tags = [ '@flounce', '@horsebrass', '@mullion', '@trivet' ]

class WitnessSay( Behave ):
    def __init__( self, agent, called ):
        Behave.__init__( self,
                         'witness_say',
                         agent,
                         template = called + ' [say/v]:',
                         salience = 0 )

    def include_for_narrator( self, narr, topobj ):
        return self.agent == narr

class MyTake( Configure ):
    def __init__( self, actor, thing, is_murdery = False ):
        Configure.__init__( self,
                            'take',
                            actor,
                            direct = thing,
                            new = ('in', actor),
                            template = '[' + actor + '/s] [pick/v] up [direct/o]' )
        self.is_murdery = is_murdery

class MyDrop( Configure ):
    def __init__( self, actor, thing, room, is_murdery = False ):
        Configure.__init__( self,
                            'drop',
                            actor,
                            direct = thing,
                            new = ('in', room ),
                            template = '[' + actor + '/s] [put/v] down [direct/o]' )
        self.is_murdery = is_murdery

class MyAfterArrive( Behave ):
    def __init__( self, agent, room ):
            Behave.__init__( self,
                             'afterarrive',
                             agent,
                             target = room,
                             template='[agent/s] [come/v] into [direct/o]',
                             salience = 0 )

    def include_for_focalizer( self, witness, topobj ):
        return ( ( self.agent == topobj or topobj in self.moving_with )
                 and witness != self.agent )

class MyKill( Behave ):
    def __init__( self, agent, victim ):
        Behave.__init__( self,
                         'kill',
                         agent,
                         target=victim,
                         template = '[agent/s] crushed [@indigo/o] under the nose of Sir Horace, who quickly died!' )
        self.is_murdery = True

class MyRemark( Behave ):
    def __init__( self, agent, tmplt, involves = [] ):
        Behave.__init__( self,
                         'say',
                         agent,
                         template = tmplt )
        self.involves = involves

    def include_for_focalizer( self, witness, topobj ):
        return self.agent == witness or self.agent == topobj or topobj in self.involves

class MyExamineNewRoom( Sense ):
    def __init__( self, agent, room ):
            Sense.__init__( self,
                            'examine', 
                            agent,
                            modality='sight',
                            direct=room,
                            salience = 0 )
            self.cause = ':' + str(self.id) + ':'
            self.is_murdery = False

    def include_for_focalizer( self, witness, topobj ):
        # only a person talking about the room
        return self.agent == witness and topobj == self.direct

class MyEnter( Configure ):
    def __init__( self, actor, goal, **kw ):
        Configure.__init__( self,
                            'enter',
                            actor,
                            template='[agent/s] [come/v] into [indirect/o]',
                            direct=actor, new=('in', goal),
                            salience = 0.0 )
        self.is_murdery = False

    def entails( self, world ):
        # adapted from superclass to use our own entailed examine
        actions = []
        if len (self.failed) > 0:
            return actions
        if ( world.item[self.direct].actor and
             not self.old_parent == self.new_parent and
             not self.new_parent == '@cosmos' ):
            room = self.new_parent

            aa = MyAfterArrive( self.direct, room )
            aa.moving_with = self.moving_with
            aa.is_murdery = self.is_murdery
            actions.append( aa )

            ex = MyExamineNewRoom( self.direct, room )
            ex.is_murdery = self.is_murdery
            actions.append( ex )

        actions += self.enlightened
        return actions

    def include_for_focalizer( self, witness, topobj ):
        return False

class MyLeave( Behave ):
    def __init__( self, actor, direction, destroom, moving_with = [], is_murdery = False ):

        # destroom hack because I can't work out how to make people know about all the rooms

        tmplt = '[agent/s] [go/v] into ' + tag_to_called[ destroom ]

        Behave.__init__( self,
                         'leave',
                         actor,
                         direction = direction,
                         template = tmplt,
                         salience = 0 )
        self.moving_with = moving_with
        self.is_murdery = is_murdery

    def entails( self, world ):
        # adapted from superclass version to use our own enter event
        actions = []
        if len(self.failed) > 0:
            return actions

        room = world.room_of(self.agent)
        if room.exit(self.direction) is not None:
            goal = room.exits[self.direction]

            new = MyEnter( self.agent, goal )
            new.is_murdery = self.is_murdery
            new.moving_with = self.moving_with

            actions.append(new)
        return actions

    def include_for_focalizer( self, witness, topobj ):
        return self.agent == topobj or self.agent == witness or topobj in self.moving_with

class MurderEndMarker( Behave ):
    def __init__( self, suspect ):
        Behave.__init__( self,
                         'murderend',
                         suspect,
                         template = 'XXX internal murder end marker action XXX',
                         salience = 0 )

    def include_for_focalizer( self, witness, topobj ):
        return False

init_murder_actions = [
    WitnessSay( '@horsebrass', 'Captain Horsebrass' ),
    WitnessSay( '@flounce', 'Miss Flounce' ),
    WitnessSay( '@trivet', 'Trivet' ),
    WitnessSay( '@mullion', 'Lady Mullion' ),

    # make them know about the initial stuff
    MyExamineNewRoom( '@horsebrass', '@library' ),
    MyExamineNewRoom( '@flounce', '@library' ),
    MyExamineNewRoom( '@mullion', '@library' ),
    MyExamineNewRoom( '@trivet', '@hall' )
]

murder_actions = [
    # horsebrass shows people the indigo
    MyTake( '@horsebrass', '@indigo' ),

    MyRemark( '@horsebrass',
              """[agent/s] [show/v] everyone [@indigo/o],
              explaining that its fumes could kill a man in seconds""" ),

    # everyone moves to drawing room, horsebrass last
    MyLeave( '@mullion', 'east', '@drawingroom' ),
    MyLeave( '@flounce', 'east', '@drawingroom' ),

    MyLeave( '@horsebrass', 'east', '@drawingroom', moving_with = ['@indigo'] ),

    MyDrop( '@horsebrass', '@indigo', '@drawingroom' ),

    # mullion moves the indigo into the middle room, comes back
    MyRemark( '@mullion',
              """[agent/s] [say/v] that [agent/s] [is/v] nervous looking at [@indigo/o],
              and would [put/v] it back in the library""" ),

    MyTake( '@mullion', '@indigo' ),
    MyLeave( '@mullion', 'west', '@library', moving_with = ['@indigo'] ),
    MyDrop( '@mullion', '@indigo', '@library' ),
    MyLeave( '@mullion', 'east', '@drawingroom' ),

    # flounce leaves next
    MyRemark( '@flounce',
              """[agent/s] [say/v] that [agent/s] would look for a book in the library""" ),
    MyLeave( '@flounce', 'west', '@library' ),

    # flounce grabs indigo, kills people
    MyTake( '@flounce', '@indigo', is_murdery = True ),
    MyLeave( '@flounce', 'north', '@study', moving_with = ['@indigo'], is_murdery = True ),
    MyKill( '@flounce', '@victim' ),
    MyDrop( '@flounce', '@indigo', '@study', is_murdery = True ),

    # trivet walks past
    MyTake( '@trivet', '@salver' ),
    MyLeave( '@trivet', 'east', '@library' ),
    MyDrop( '@trivet', '@salver', '@library' ),
    MyLeave( '@trivet', 'east', '@drawingroom' ),

    # flounce back to drawing room
    MyLeave( '@flounce', 'south', '@library', is_murdery = True ),
    MyLeave( '@flounce', 'east', '@drawingroom' ),

    MyRemark( '@trivet',
              """[@trivet/s] [ask/v] if anyone required a drink""" ),

    # horsebrass back out
    MyLeave( '@horsebrass', 'west', '@library' ),
    # and horsebrass find body
    MyLeave( '@horsebrass', 'north', '@study' ),

    MyRemark( '@horsebrass',
              """Sir Horace had been killed with [@indigo/o]!""",
              involves = [ '@indigo', '@victim' ] )

] + [ MurderEndMarker( s ) for s in suspect_tags ]

init_interrogate_actions = [
    MyLeave( '@horsebrass', 'south', '@library' ),
    MyLeave( '@horsebrass', 'west', '@hall' ),
    MyLeave( '@horsebrass', 'west', '@morningroom' ),

    MyLeave( '@trivet', 'west', '@library' ),
    MyLeave( '@trivet', 'west', '@hall' ),
    MyLeave( '@trivet', 'west', '@morningroom' ),

    MyLeave( '@mullion', 'west', '@library' ),
    MyLeave( '@mullion', 'west', '@hall' ),
    MyLeave( '@mullion', 'west', '@morningroom' ),

    MyLeave( '@flounce', 'west', '@library' ),
    MyLeave( '@flounce', 'west', '@hall' ),
    MyLeave( '@flounce', 'west', '@morningroom' )
]

initial_actions = init_murder_actions + murder_actions + init_interrogate_actions + [
    Sense('see', '@holmes', direct='@morningroom', modality='sight'),
]
